/*
Revision history

Date 		What									Who  Version
2005-12-20	First version. Hardware works after change of CPU...Puh!		Mce  0.1
2006-01-05	Serial port 2 for NMEA working at 4800 baud				Mce  0.2
2006-01-10	I2C Works fine, A/D convert code fixed					Mce  0.3
2006-01-13	EEPROM code fixed							Mce  0.4

Comments;
TODO Sealtalk bit bang code or should I go for NMEA?
None - so far...

*/

//#define MCS51REG_ENABLE_WARNINGS
#define MICROCONTROLLER_DS89C420
#include <sdcc_reg420.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <mcs51reg.h>
#include "ser_ir_2.h"

//#define DEBUG  1


// This is the HARDWARE pins used fo I2C
#define ClutchRelay	P0_0	// High = Relay/Clutch on
#define SCL  		P0_1
#define SDA  		P0_2

#define ADint		P1_4	// Interrupt from Analog2Digital conv. MAX1361

#define SeatalkIn	P2_0
#define SeatalkOut	P2_1

#define StearLeft	P2_2
#define StearRight  	P2_3

#define READ 		1
#define WRITE		0
#define TRUE 		1
#define FALSE		0
#define ON   		1
#define OFF   		0


// I2C Hardware Device adresses
#define EEPROM 		0xA0 	// M24C32W6  1010abcw, a,b,c is hardwire, w = r/w bit.
#define MAX1362		0x68	// MAX1362EUB -40C to +85C 10 MAX 0110100/0110101 4.5 to 5.5V



// Prototypes and definitions for I2C
unsigned char 	_i2c_error;      // bit array of error types
void          	_I2CBitDly( void );
void          	_I2CSCLHigh( void );
void           	I2CSendAddr( unsigned char, unsigned char );
void           	I2CSendByte( unsigned char );
unsigned char 	_I2CGetByte( unsigned char );
void           	I2CSendStop( void );
#define 	I2CGetByte()      _I2CGetByte(0)
#define 	I2CGetLastByte()  _I2CGetByte(1)
#define 	I2CERR_SCL_LOW 		0x02
#define 	I2CERR_NO_ACK 		0x01


// Prototypes and definitions for EEPROM routines
unsigned char EpromByteRead( int );
void EpromByteWrite( int adr, unsigned char c );
int EpromReadInt( int );
void EpromWriteInt( int, int );
void EpromStringWrite( int adr, const char *s );

// Other prototypes and definitions
void timer0_isr(void) interrupt 1;
void out2hex( unsigned char );
void delay_mS( int );
void delay_uS( int );
void clock_init( void );
void get_time(void);
void DumpEprom(void);
int rudderpos( void );
int ConvDirection( int dir );

// GLOBAL VARIABLES %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
unsigned char hours;
unsigned char minutes;
unsigned char seconds;
volatile bit Time_Changed;

//############################################################################

void putchar(char c)
{
	ser_putc(c);
}


char getchar(void)
{
	return(ser_getc());
}


void clrscr( void )
{
	printf_fast("\x1b[2J");
}



void gotoxy(unsigned char x, unsigned char y )
{
	printf_fast("\x1b[%d;%dH",y,x);
}


/*****************************************************************************/

volatile unsigned char s_hours;		// Special vars for ISR, don't use these: See get_time below
volatile unsigned char s_minutes;
volatile unsigned char s_seconds;

void clock_init()
{
	s_hours = s_minutes = s_seconds = 0;	// zero hours, minutes, seconds
	TR0  = 0;			// make sure timer 0 is stopped
	TF0  = 0;			// clear the overflow flag
	TMOD &= 0xF0;			// set to mode 0 (timer1 unchanged)
	TL0  = TH0 = 0;			// clear the timer 0 value
	TR0  = 1;			// start the timing
	Time_Changed = 0;
	ET0  = 1;                       // Enable Timer 0 Interrupts
}

void timer0_isr( void ) interrupt 1
{
	static unsigned char timerticks = 126;

	if ( --timerticks == 0 )
	{
		timerticks = 126;
		if ( ++s_seconds == 60 )
		{
			s_seconds = 0;
			if ( ++s_minutes == 60 )
			{
				s_minutes = 0;
				if ( ++s_hours == 24 )
				{
					s_hours = 0;
				}
			}

		}
		Time_Changed = 1;
	}
}


void get_time(void) //critical
{
	ET0 = 0;		// temporarily disable interrupt
	hours = s_hours;
	minutes = s_minutes;
	seconds = s_seconds;
	Time_Changed = 0;
	ET0 = 1;		// reenable interrupt ASAP
}

/*****************************************************************************/


// Delay routines. Made for SDCC on a DS89C420.
void delay_mS( int ms ) //time in mS
{
	for( ; ms; ms-- ) // this loop time is NOT included in calculations below
		delay_uS( 99 );
}


// Delay routines. Made for SDCC on a DS89C420.
void delay_uS( int us ) //time in uS
{
	for( ; us; us-- ) // this loop time is NOT included in calculations below
	{
	_asm
	nop
	nop
	nop
	nop
	_endasm;
	}
}



// convert lower nibble of x to an ascii char and print it to screen
void outhex( unsigned char x )
{
	x &= 0x0f;
	if ( x > 9 ) x += 7;
	putchar( '0' + x );
}


// convert x to two ascii chars and print them to screen
void out2hex( unsigned char x )
{
	outhex( x >> 4);
	outhex( x );
}

/*
unsigned char in2hex( void )
{
	unsigned char c1, c2;

	c1 = ser_getc();
	c1 -= '0';
	if ( c1 > 9 )
		c1 -= 7;
	c1 <<= 4;

	c2 = ser_getc();
	c2 -= '0';
	if ( c2 > 9 )
		c2 -= 7;

	return( c1 + c2 );
}
*/



void _I2CSCLHigh( void ) {      // set SCL high, and wait for it to go high

	int err1 = 0;	// Could be "register" but then too fast?

	SCL = 1;
	while ( ! SCL ) { // wait for SCL to go high, could be low due to "clock stretching" from a slow slave.
		err1++;
		if ( err1 > 250 ) {
			_i2c_error |= I2CERR_SCL_LOW;     // SCL probably stuck, something's holding it down
			return;
		}
	}
	delay_uS(5);
}



void _I2CSCLlow( void ) {
	SCL = 0;
	delay_uS(5);
}



void _I2CSDAlow( void ) {
	SDA = 0;
	delay_uS(5);
}



void _I2CSDAhigh( void ) {
	SDA = 1;
	delay_uS(5);
}




void I2CSendByte(unsigned char bt) {
	int i;

	for ( i = 0; i < 8; i++) {
		if ( bt & 0x80 )
			SDA = 1;      // send each bit, MSB first
		else
			SDA = 0;
		_I2CSCLHigh();
		_I2CSCLlow();
		bt <<= 1;
	}
	SDA = 1;			// listen for ACK
	_I2CSCLHigh();
	if (SDA) {
		_i2c_error |= I2CERR_NO_ACK;          // ack didn't happen, may be nothing out there

#if DEBUG
		ser_putc('A');
		ser_putc('C');
		ser_putc('K');
#endif

	}
	_I2CSCLlow();
}



void I2CSendAddr(unsigned char address, unsigned char rd) {
	_I2CSCLHigh();
	_I2CSDAlow();
	_I2CSCLlow();
	I2CSendByte( address + rd );  // send address byte
}



unsigned char _I2CGetByte(unsigned char lastone) { // lastone == 1 for last byte

	int i;
	unsigned char res;

	res = 0;
	for ( i = 0; i < 8; i++) {		// each bit at a time, MSB first
		_I2CSCLHigh();
		res *= 2;
		if ( SDA )  res++;
		_I2CSCLlow();
	}
	SDA = lastone;             // send ACK according to 'lastone'
	_I2CSCLHigh();
	_I2CSCLlow();
	SDA = 1;
	return( res );
}


void I2CSendStop( void ) {
	_I2CSDAlow();
	_I2CSCLHigh();
	_I2CSDAhigh();
}

// read one byte from the specified address
unsigned char i2c_read_1( unsigned char address ) {
	I2CSendAddr( address, READ );
	return( I2CGetLastByte() );
}


// write one byte to the specified address
void i2c_write_1( unsigned char address, unsigned char d1 ) {
	I2CSendAddr( address, WRITE );
	I2CSendByte( d1 );
	I2CSendStop();
}



void i2c_write_2( unsigned char address, unsigned char d1 , unsigned char d2 ) {
	I2CSendAddr( address, WRITE );
	I2CSendByte( d1 );
	I2CSendByte( d2 );
	I2CSendStop();
	delay_mS(10);
}



//#define M24C01  1 //  128 x 8
//#define M24C02  2 //  256 x 8
//#define M24C04  4 //  512 x 8 This and all bigger EEPROMS below needs different addressing.
//#define M24C08  8 // 1024 x 8
//#define M24C16 16 // 2048 x 8
//#define M24C32 32 // 4096 x 8
//#define M24C64 64 // 8192 x 8

unsigned char EpromByteRead( int adr ) // Read byte from EEPROM
{
	unsigned char x;

	I2CSendAddr( EEPROM, WRITE );
	I2CSendByte( (unsigned char) adr & 0xff);
	I2CSendAddr( EEPROM, READ );
	x = I2CGetLastByte();
	I2CSendStop();
	return(x);
}

int EpromReadInt( int adr ) // Read Integer Word (2 bytes) from EEPROM
{
	unsigned char a,b;


	a = EpromByteRead( (unsigned char) adr & 0xff);
	b = EpromByteRead( ( (unsigned char) adr &0xff ) + 1 );
	return ( (a * 256) + b);
}


void EpromByteWrite( int adr, unsigned char c )
{
	i2c_write_2(EEPROM, (unsigned char) adr, c);
	delay_mS(10);
}

void EpromWriteInt( int adr, int d1 )
{
	I2CSendAddr( EEPROM, WRITE );
	I2CSendByte( (unsigned char) adr );
	I2CSendByte( d1 >> 8 );
	I2CSendByte( d1 );
	I2CSendStop();
	delay_mS(10);
}


void EpromStringWrite( int adr, const char *s )
{
	int i;

	for ( i = 0; s[i] != '\0'; i++ )
		EpromByteWrite( s[i], (unsigned char) adr + i );
}


void DumpEprom( void )
{
	int i = 0;
	int j = 0;

	ser_puts("\n\r00");
	out2hex( (unsigned char) i );
	ser_puts(" ");
	for ( i = 0; i < 128; i++ ) {
		out2hex( EpromByteRead(i) );
		ser_puts(" ");
		if ( j == 7 )
			ser_puts(": ");
		if ( j++ == 15 ) {
			j = 0;
			ser_puts("\n\r00");
			out2hex( i + 1 );
			ser_puts(" ");
		}
	}
	printf_fast("\n\r");
}


// read rudder position convert to integer
int rudderpos( void )
{
	int x;

	I2CSendAddr( MAX1362, READ );
	x  = ( I2CGetByte() & 0x03 ) * 256; // Read maximum 10 bits from ADC!
	x += I2CGetLastByte();
	I2CSendStop();
	return( x );
}



#define LEFT  1
#define STOPP 2
#define RIGHT 3

void Steer( unsigned char Dir )
{
	if ( Dir == LEFT )
	{
		StearRight = OFF;
		StearLeft  = ON;
	}

	if ( Dir == STOPP )
	{
		StearRight = OFF;
		StearLeft  = OFF;
	}

	if ( Dir == RIGHT )
	{
		StearLeft  = OFF;
		StearRight = ON;
	}
}

void Clutch ( unsigned char Dir )
{
	StearRight = OFF;
	StearLeft  = OFF;

	if ( Dir == ON )
		ClutchRelay = ON;
	else
		ClutchRelay = OFF;
}


int ConvDirection( int dir )
{
	if ( dir > 180 )
	{
		return( (-180) + ( dir - 180) );
	}
	else
	{
		return( dir);
	}
}


int ErrorCalc( int Cmp, int Hdg )
{
	int D18, x;

	D18 = ConvDirection( Cmp );  // Convert compass degrees to +-180 values...
	x = ( D18 - Hdg ) & 0x7fff; // ABS ???
	printf_fast("(%d (%d))",(D18 - Hdg), x );
	if ( x > 180 )
	{
		return( ( D18 - Hdg ) + 360 );
	}
	else
	{
		return( D18 - Hdg );
	}
}


int _sdcc_external_startup(void)
{
	// Enable internal SRAM and Initiate RAM
	PMR |= 0x01;
	return(0);
}

///////////////////////////////////////////////////////////////////////////////////////
void main( void )
{
	int Rudder, Compass, Heading, Error;
	unsigned char Mode;
	unsigned char a, c;
	xdata unsigned char buf[12];



	// Investigate reason for this boot, if watchdog then report error...
/*	if ( (WDCON & 0x40) == 0)
	{
		printf("\r\nWDCON = %02x\r\n", WDCON);
		TA = 0xAA;
		TA = 0x55;
		POR = 1;	// Clear "Power On Reset" flag
	}
*/

/*	// Enable Watchdog timer, interrupt
	CKCON |= 0xc0;	// Set Watchdog timer prescaler to 2^24
	TA = 0xAA;
	TA = 0x55;
	EWT = 1;
	EWDI = 1;
*/


	ClutchRelay = OFF;
	StearLeft = OFF;
	StearRight = OFF;

	SCL = 1;				// Init I2C lines
	SDA = 1;

	ser_init(); 				// Start Serial ports
	ser2_init();

	_i2c_error = 0;				// Initialize the A/D chip
	i2c_write_2( MAX1362, 0x07, 0x82 );
	printf_fast("I2C Init: %x\r\n",_i2c_error);

	clock_init();				// Start timekeeper

#define STANDBY   1
#define SETUP     2
#define AUTO      3
#define AUTOTACK  4

	Mode = STANDBY;

	printf_fast("Hello!\r\n");


	while(1)
	{
		if ( (Mode == STANDBY ) && ( Time_Changed == 1 ))
		{
			get_time();
			Rudder = rudderpos();
			printf_fast("%02u:%02u:%02u Rudder %d Heading %d\n\r", hours, minutes, seconds, Rudder, Compass);
		}

		if (ser2_kbhit() > 0 )
		{
			if ( ser2_getc() == '$' )	// If we never gets "$" then no NMEA data!
			{
				ser2_gets (buf, 12);
				if ( strcmp ( strtok( buf, ",") , "IIHDM" ) == 0 )
				{
					Compass = (unsigned char) atoi( strtok( NULL, ",") );
				}
			}
		}


		if ( (Mode == AUTO) && (Time_Changed == 1 ))
		{
			get_time();
			Rudder = rudderpos();
			printf_fast("1 AUTO   R: %d, HDM: %d, Heading: %d, ",  Rudder, Compass, Heading );
			Error = ErrorCalc( Compass, Heading);
			printf_fast("Error: %d degrees\n\r", Error );
		}


		if ( ser_kbhit() > 0 )
		{
			c = ser_getc();
			if ( c == '1' )  // Clutch on
			{
				Clutch( ON ) ;
				printf_fast("Clutch ON\r\n");
			}

			if ( c == '2' )
			{
				Clutch( OFF );
				printf_fast("Clutch OFF\r\n");
			}

			if ( c == '3' )
			{
				Steer(LEFT);
				printf_fast("going Left\r\n");
			}

			if ( c == '4' )
			{
				Steer(STOPP);
				printf_fast("stop\r\n");
			}

			if ( c == '5' )
			{
				Steer(RIGHT);
				printf_fast("going Right\r\n");
			}


			if ( c == 'c' )
			{

				Mode = SETUP;

				ser_puts("Rudder midship: Press Enter ");
				a = ser_getc();
				ser_puts("\n\r");
				if (a == 'q') continue;

				Rudder = rudderpos();
				EpromWriteInt( 0, Rudder );
				printf_fast(" %d \r\n", EpromReadInt(0) );


				ser_puts("Rudder right-turn: Press Enter ");
				a = ser_getc();
				ser_puts("\n\r");
				if (a == 'q') continue;

				Rudder = rudderpos();
				EpromWriteInt( 2, Rudder );
				printf_fast(" %d \r\n", EpromReadInt(2) );

				ser_puts("Rudder left-turn: Press Enter ");
				a = ser_getc();
				ser_puts("\n\r");
				if (a == 'q') continue;

				Rudder = rudderpos();
				EpromWriteInt( 4, Rudder );
				printf_fast(" %d \r\n", EpromReadInt(4) );


				printf_fast("Left=%d ", EpromReadInt(4) );
				printf_fast("Center=%d ", EpromReadInt(0) );
				printf_fast("Right=%d\r\n\n", EpromReadInt(2) );

				ser_puts("Press any key...");
				a = ser_getc();
				ser_puts("\r\n");

				Mode = STANDBY;
			}


			if ( c == 'd' )
			{
				DumpEprom( );
			}

			if ( c == 'z' )
			{
				printf_fast(" ReadInt= %d %d %d", EpromReadInt(0), EpromReadInt(2), EpromReadInt(4));

			}


			// Enter the fameus AUTO mode
			// THIS is the course wanted!!! <<<<==========

			if ( c == 'a' )
			{
				Mode = AUTO;
				Heading = ConvDirection( Compass );
				// Clutch ON
			}


			if ( c == 's' )
			{
				Mode = STANDBY;
				ClutchRelay = OFF;
			}
		}
	}
}

/*
	 rudder gain  (1-9) [2]                                       1
	 rudder limit  (10-40) [30]                                   3
	 off course limit  (15-40) [20]                               6

	 rudder damping  (1-9) [2]                                    B

	 variation: (full degrees)(-30 to +30) [0]                    C
	 auto adapt: 0=Off,1=North,2=South [1]                        D
	 auto adapt latitude (0-80) [0]                               E
	 auto release (only for stern drive) ON/OFF                   F
	 rudder alignment (-7 to +7) [0]                             10
	 Wind Trim (Wind Response) (1-9) [5] (only for sail)         11
	 Response  (1-9) [5]                                         12
	 Boat type:1=displ,2=semi-displ,3=plan,4=stern,5=work,6=sail 13
	 Cal Lock:  0=OFF, 1=ON [0]                                  15
	 Auto Tack Angle (40-125) [100] (only for sail)              1d
*/