/*-------------------------------------------------------------------------
  ser_ir.c - source file for serial routines

  Written By - Josef Wolf <jw@raven.inka.de> (1999)

	 This program is free software; you can redistribute it and/or modify it
	 under the terms of the GNU General Public License as published by the
	 Free Software Foundation; either version 2, or (at your option) any
	 later version.

	 This program is distributed in the hope that it will be useful,
	 but WITHOUT ANY WARRANTY; without even the implied warranty of
	 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	 GNU General Public License for more details.

	 You should have received a copy of the GNU General Public License
	 along with this program; if not, write to the Free Software
	 Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

	 In other words, you are welcome to use, share and improve this program.
	 You are forbidden to forbid anyone else to use, share and improve
	 what you give them.   Help stamp out software-hoarding!

-------------------------------------------------------------------------*/

/* This file implements a serial interrupt handler and its supporting
* routines. Compared with the existing serial.c and _ser.c it has
* following advantages:
* - You can specify arbitrary buffer sizes (umm, up to 255 bytes),
*   so it can run on devices with _little_ memory like at89cx051.
* - It won't overwrite characters which already are stored in the
*   receive-/transmit-buffer.
* - It checks receiver first to minimize probability for overruns
*   in the serial receiver.
*/

/* BUG: those definitions (and the #include) should be set dynamically
* (while linking or at runtime) to make this file a _real_ library.
*/
//#define MCS51REG_ENABLE_WARNINGS
#define MICROCONTROLLER_DS89C420
//#include <mcs51reg.h>
#include <sdcc_reg420.h>
#include <ser_ir_2.h>

#define XBUFLEN 8
#define RBUFLEN 8

#define X2BUFLEN 18
#define R2BUFLEN 18


xdata volatile unsigned char rbuf[RBUFLEN], xbuf[XBUFLEN];
xdata volatile unsigned char rcnt, xcnt, rpos, xpos;
xdata volatile unsigned char busy;
xdata volatile unsigned char r2buf[R2BUFLEN], x2buf[X2BUFLEN];
xdata volatile unsigned char r2cnt, x2cnt, r2pos, x2pos;
xdata volatile unsigned char b2usy;

/*
void ResetWDT( void )
{
	// clear watchdog timer and interruptflag.
	TA = 0xAA;
	TA = 0x55;
	WDCON |= 0x03;
}
*/
// WDCON.0 RWT Reset Watchdog timer
// WDCON.1 EWT Watchdog Reset Enable
// WDCON.3 WDIF Watchdog Interrupt Flag
/*
void WatchDog_ISR ( void ) interrupt 12
{
	TA = 0xAA;
	TA = 0x55;
	WDCON |= 0x03;
}
*/


/*
 RCAP2H,RCAP2L = 65536 - (Osc Freq / (32 * Baudrate) )
 65536 - (3579545 / ( 32 * 9600 ) ) = 65524,347835286458333333333333333
 witch gives FFF4 hex.
 1.843 200 @  9600 baud = FFFA
11.059 200 @  4800 baud = FFB8
11.059 200 @  9600 baud = FFDC
11.059 200 @ 19200 baud = FFEE
11.059 200 @ 57600 baud = FFFC
*/


// Serial Port 0 & Timer 2 initialisation. ENABLES GLOBAL INTERUPTS
void ser_init (void) {
	EA    = 0;			// Stop Global Interrupts
	ES0   = 0; 			// disable serial channel 0 interrupt
	TR_2  = 0; 			// stop timer 2
	T2CON = 0x30; 			// b0011_0100 Timer2 is Rx & Tx baudclock
	TL2   = RCAP2L = 0xEE;		// Timer delay value
	TH2   = RCAP2H = 0xFF;
	TF_2  = 0; 			// clear overflow flag
	SCON0 = 0x50; 			// Serial port in mode 1, baudrate from timer 2, enable receiver, clear RI and TI
	TR_2  = 1;			// Start Timer
	RI_0  = 0;			// clear "pending" interrupts
	TI_0  = 0;
	ES0   = 1; 			// enable serial channel 0 interrupts
	EA    = 1;			// Set Global Interrupt enable

	rcnt = xcnt = rpos = xpos = 0;  // init buffers
   	busy = 0;
}

void ser_handler (void) interrupt 4 {
   	if (RI_0) {
	   	RI_0 = 0;
	   	/* don't overwrite chars already in buffer */
	   	if (rcnt < RBUFLEN)
		   	rbuf [(rpos+rcnt++) % RBUFLEN] = SBUF0;
   	}
   	if (TI_0) {
		   TI_0 = 0;
		   if (busy = xcnt) {   /* Assignment, _not_ comparison! */
			   xcnt--;
			   SBUF0 = xbuf [xpos++];
			   if (xpos >= XBUFLEN)
				   xpos = 0;
		   }
   	}
}

//void putchar (unsigned char c) {
void ser_putc (unsigned char c) {
   	while (xcnt >= XBUFLEN) /* wait for room in buffer */
	   ;
   	ES0 = 0;
   	if (busy) {
		   xbuf[(xpos+xcnt++) % XBUFLEN] = c;
   	} else {
		   SBUF0 = c;
		   busy = 1;
   	}
   	ES0 = 1;
}

unsigned char ser_getc (void) {
   	xdata unsigned char c;

   	while ( !rcnt )   /* wait for character */
	   ;
   	ES0 = 0;
   	rcnt--;
   	c = rbuf[rpos++];
   	if ( rpos >= RBUFLEN )
		   rpos = 0;
   	ES0 = 1;
   	return( c );
}


unsigned char ser_kbhit (void) {
	return( rcnt );
}


#pragma save
#pragma noinduction
void ser_puts ( unsigned char *s) {
   	xdata unsigned char c;

   	while (c=*s++) {
	   	if (c == '\n') ser_putc ('\r');
	   	ser_putc (c);
   	}
}
#pragma restore


/*
void ser_gets (unsigned char *s, unsigned char len) {
   	unsigned char pos, c;

   	pos = 0;
   	while (pos < len) {
	   	c = ser_getc ();
	   	ser_putc( c );			// Echo chars back
	   	if (c == '\r') continue;        // discard CR's
	   	s[pos++] = c;
	   	if (c == '\n') break;           // NL terminates
   	}
   	s[pos] = '\0';
}


unsigned char ser_can_xmt (void) {
   	return XBUFLEN - xcnt;
}


unsigned char ser_can_rcv (void) {
   	return rcnt;
}

*/

/**************************************************************************************
 RCAP2H,RCAP2L = 256 - (Osc Freq / (32 * Baudrate) )

*/


#define TCLK 11059200UL
// TH1 = (unsigned char) (256 - (TCLK / (16L * 12L * 4800)));

// Serial Port 1 & Timer 1 initialisation. ENABLES GLOBAL INTERUPTS
void ser2_init (void) {
	EA    = 0;			// Stop Global Interrupts
	ES1   = 0; 			// disable serial channel 1 interrupt
	TR1   = 0; 			// stop timer 1

	TMOD &= ~0xF0;			// clear timer 1 mode bits
	TMOD |= 0x20;           	// put timer 1 into MODE 2
	TL1   = 0xFF;
	TH1   = 0xFA;			// 4800 baud SMOD=0
	// TH1   = 0xFD;			// 9600 baud SMOD=0
    	SCON1 = 0x50;           	// Set SCI_1 to 8N1, Rx enabled
	RI_1  = 0;			// clear "pending" interrupts
	TI_1  = 0;
	TR1   = 1; 			// start timer 1
	ES1   = 1; 			// enable serial channel 1 interrupt
	EA    = 1;			// Set Global Interrupt enable

	r2cnt = x2cnt = r2pos = x2pos = 0;  // init buffers
   	b2usy = 0;
}



void ser2_handler (void) interrupt 7
{
   	if (RI_1) {
	   	RI_1 = 0;
	   	/* don't overwrite chars already in buffer */
	   	if (r2cnt < R2BUFLEN)
		   	r2buf [(r2pos + r2cnt++) % R2BUFLEN] = SBUF1;
   	}
   	if (TI_1) {
		   TI_1 = 0;
		   if (b2usy = x2cnt) {   /* Assignment, _not_ comparison! */
			   x2cnt--;
			   SBUF1 = x2buf [x2pos++];
			   if (x2pos >= X2BUFLEN)
				   x2pos = 0;
		   }
   	}
}

/*
void ser2_putc (unsigned char c)
{
   	while (x2cnt >= X2BUFLEN) // wait for room in buffer
	   ;
   	ES1 = 0;
   	if (b2usy) {
		   x2buf[(x2pos + x2cnt++) % X2BUFLEN] = c;
   	} else {
		   SBUF1 = c;
		   b2usy = 1;
   	}
   	ES1 = 1;
}

*/


unsigned char ser2_getc (void)
{
//unsigned char getchar (void)
   	xdata unsigned char c;

   	while ( !r2cnt )   /* wait for character */
	   ;
   	ES1 = 0;
   	r2cnt--;
   	c = r2buf[r2pos++];
   	if ( r2pos >= R2BUFLEN )
		   r2pos = 0;
   	ES1 = 1;
   	return( c );
}


void ser2_gets (unsigned char *s, unsigned char len)
{
   	unsigned char pos, c;

   	pos = 0;
   	while (pos < len) {
	   	c = ser2_getc ();
	   	// ser_putc( c );			// Echo chars back
	   	if (c == '\r') continue;        // discard CR's
	   	s[pos++] = c;
	   	if (c == '\n') break;           // NL terminates
   	}
   	s[pos] = '\0';
}


/*
#pragma save
#pragma noinduction
void ser2_puts ( unsigned char *s) {
   	xdata unsigned char c;

   	while (c=*s++) {
	   	if (c == '\n') ser2_putc ('\r');
	   	ser2_putc (c);
   	}
}
#pragma restore
*/

unsigned char ser2_kbhit (void) {
	return( r2cnt );
}

